


/******************************************************************************************
 *                                                                                        *
 * Examples : RTC+TEMP Sensor(DS3232)  For "ET-BASE PIC8722 (ICD2)"                       *
 *                                                                                        *
 *          :  Interface I2C (Use I/O Pin)                                                *                                                                                        
 *                                                                                        *
 ******************************************************************************************
 *                                                                                        *
 * Target MCU        : PIC18F8722                                                         *
 *                   : X-TAL : 10 MHz                                                     *
 *                   : CPU Clock = 40 MHz (x4)                                            *                                                                  *
 * Editor-Compiler   : CCS Version 4.120                                                  *
 * Create By         : Mr. Sittiphol Yooyod (WWW.ETT.CO.TH)                               *
 * Last Update       : 25/December/2011                                                   *
 *                                                                                        *
 * DS3232            : Interface I2C(ID:1101000X = 0xD0(Wr),0xD1(Rd) )                    *          
 *                                                                                        *
 * Port Interface    :                                                                    *
 *                                                                                        *
 *             LCD   :                                                                    *
 *                     RH1     = RS(OUT)                                                  *
 *                     RH2     = R/W(OUT)                                                 *
 *                     RH3     = EN(OUT)                                                  *  
 *                     RH4-RH7 = Data 4 bit high(Out)                                     *
 *                                                                                        * 
 *        5V                                                                              *
 *      --+--                                                                             *
 *        |                                                                               *
 *        +-----------------+                                                             *
 *        |                 |                                                             *
 *       ---                |                                                             *
 *       |\|                |                                                             *
 *    VR |/|<---------------|--+                                                          *
 *   10K ---                |  |  RH1 RH2 RH3            RH4 RH5 RH6 RH7                  *
 *        |                 |  |   ^   ^   ^              ^   ^   ^   ^                   *
 *        |                 |  |   |   |   |              |   |   |   |                   *
 *        +--------------+--|--|---|---|---|--+--+--+--+  |   |   |   |                   *
 *      __|__            |  |  |   |   |   |  |  |  |  |  |   |   |   |                   *
 *       ---             |  |  |   |   |   |  |  |  |  |  |   |   |   |                   *
 *        -             1| 2| 3|  4|  5| 6 | 7| 8| 9|10|11|12 | 13| 14|                   *
 *                     __|__|__|___|___|___|__|__|__|__|__|___|___|___|_                  *
 *                    |  G Vcc Vo RS  R/W  E  D0 D1 D2 D3 D4  D5  D6 D7 |                 *
 *                    |    _________________________________________    |                 *
 *                    |   |                                         |   |                 *
 *                    |   |               Module                    |   |                 *
 *                    |   |         LCD 2x16 Charecter              |   |                 *  
 *                    |   |_________________________________________|   |                 *
 *                    |                                                 |                 *
 *                    |_________________________________________________|                 *
 *                                                                                        *       
 *                                                                                        *
 * I2C Interface  :   <MCU 18F8722>                        <DS3232 >                      * 
 *                                                                                        *
 *                     RA0 (SDA-I/O)      Connect           SCL                           *
 *                     RA1 (SCK-OUT)      Connect           SDA(DATA)                     *
 *                                                                                        *
 *                                                                                        *
 ******************************************************************************************/

#include <18F8722.h>
#include <stdio.h>


#fuses H4,NOLVP,NOWDT,NOPROTECT,NOSTVREN         //Setup MCU
#use delay (clock = 40000000)                    //delay Clock = (ms)

#use fast_io(ALL)                                //Stop Set direction Auto for All Port  by Compiler  

//************** Set LCD Control Pin *******************

#define ENA_HI()     output_high(PIN_H3) ;         //RH3 = ENA:1
#define ENA_LO()     output_low(PIN_H3)  ;         //RH3 = ENA:0

//************** Set I2C Control Pin *******************

#define SCL_HI()     output_high(PIN_A1) ;         //RH3 = ENA:1
#define SCL_LO()     output_low(PIN_A1)  ;         //RH3 = ENA:0

#define SDA_HI()     output_high(PIN_A0) ;         //RH3 = ENA:1
#define SDA_LO()     output_low(PIN_A0)  ;         //RH3 = ENA:0

#define SDA_IN()     set_tris_A(0x01)    ;         //Set RA0 = SDA: Input
#define SDA_OUT()    set_tris_A(0x00)    ;         //Set RA0 = SDA: Output
 

unsigned int16 ss,mn,hh,tem,dot,dd,mm,yy     ;
unsigned int8 sign                           ;

/*****************************************
 **        Function Initial Port        **
 *****************************************/

 void init_port(void)
  {
     set_tris_A(0x00)   ;    //Set RA0..RA7 = Output For I2C 
     set_tris_H(0x00)   ;    //Set RH0..RH7 = Output For LCD 
    
     OUTPUT_A(0x00)     ; 
     OUTPUT_H(0x00)     ;
  }

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  +                                                                                      +
  +                              LCD Function                                            +
  +                                                                                      +
  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/


/**********************************
 *    write instruction LCD       *
 **********************************/

void write_cmm(unsigned int8 cmm)
{
  OUTPUT_H(cmm & 0xF0 )               ;    //Sent cmm 4 bit(RH4-RH7) High , RS:RH1 = 0 ,R/W:RH2 = 0
  ENA_HI()                            ;    
  delay_us(300)                       ;
  ENA_LO()                            ;    
  delay_us(300)                       ;

  OUTPUT_H((cmm << 4) & 0xF0)         ;    //Sent cmm 4 bit(RH4-RH7) Low , RS:RH1 = 0 ,R/W:RH2 = 0
  ENA_HI()                            ;   
  delay_us(300)                       ;
  ENA_LO()                            ;   
  delay_us(300)                       ;
}

/*********************************
 *       write data lcd          *
 *********************************/

void write_data(unsigned int8 dat)
{
  
  OUTPUT_H((dat & 0xF0)|0x02)         ;    //Sent data 4 bit(RH4-RH7) High , RS:RH1 = 1 ,R/W:RH2 = 0
  ENA_HI()                            ;    
  delay_us(300)                       ;
  ENA_LO()                            ;    
  delay_us(300)                       ;

  OUTPUT_H(((dat << 4) & 0xF0)|0x02)  ;    //Sent cmm 4 bit(RH4-RH7)Low , RS:RH1 = 1 ,R/W:RH2 = 0
  ENA_HI()                            ;   
  delay_us(300)                       ;
  ENA_LO()                            ;   
  delay_us(300)                       ;
}

/******************************
 *      Clear Display LCD     *
 ******************************/

void clr_display(void)
 {
  write_cmm(0x01) ;
 }


/**********************************
 *          initial lcd           *
 **********************************/
void init_lcd(void)
{
  write_cmm(0x33);            //Command Control 
  write_cmm(0x32);
  write_cmm(0x28);
  write_cmm(0x0C);           //Set Display On/Off:Display On,Not Show Cursor and not blink Cursor
  write_cmm(0x06);           //Set Entry Mode :Increment Address,Cursor Shift Right
    
  write_cmm(0x1C);           //Set Cursor Or Display Shift : Shift Right Charecter
  write_cmm(0x28);           //Function Set: set interface data 4bit,Set Display LCD 2 Line ,Set Charecter Size 5x7
  write_cmm(0x80|0x00);      //Set Address Begin 0x00 (Line1 Charecter1)
  clr_display();
}



/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  +                                                                                      +
  +                              I2C Function                                            +
  +                                                                                      +
  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

 /**********************************************
  **                 I2C Start                **
  **********************************************/

void I2C_Start(void)
{ 
   SDA_OUT()              ;  //Set RA0 = output  
 
   SDA_HI()               ;   
   SCL_HI()               ;
   SDA_LO()               ;    
   SCL_LO()               ;
}

 /**********************************************
  **                 I2C Stop                **
  **********************************************/

void I2C_Stop(void)
{

   SDA_OUT()             ;  //Set RA0=Output      
  
   SDA_LO()              ;  
   SCL_HI()              ;     
   SDA_HI()              ;
   
}

/***********************************************************************************
 **                       Function I2C Write  Data 8 bit                          **
 **                                                                               **
 ***********************************************************************************
 **  Parameter :                                                                  **
 **              dat = Data for write 1Byte                                       **       
 ***********************************************************************************/


void I2C_WrByte(unsigned int8 dat)              
{
   unsigned int8 loop ;      


   for(loop=0;loop<8;loop++)                //Loop Write data 8 Bit 
    {                   
      //--------- write Data(SDA:RA0) ---------

      SDA_OUT()                           ;  //Set RA0=SDA= Output  

      if((dat & 0x80)== 0x80)                //Check data bit8 is 0 or 1
       {
          SDA_HI()                       ;   //Sent '1' Out Pin SDA(RA0)
       }
      else
       {
          SDA_LO()                       ;  //Sent '0' Out Pin SDA(RA0)
       }
      
      dat = dat <<1                      ;  //Shift data Next Bit

      SCL_HI()                           ;  //Clock HI       
      SCL_LO()                           ;  //Clock LO       
    } 
          
    SDA_IN()                             ;   //Set RA0=Input 

    SCL_HI()                             ;   //Start ACK Clock
    while(input(PIN_A0)== 0x01){;}           //Check ACK:RA0 = 0 Out Loop
    SCL_LO()                             ;   //End ACK Clock
       
}



/************************************************************************ 
 **                   Function Write RTC DS3232                        **
 ** Parameter :                                                        **
 **            addr  = Register Address for write(1Byte)               **
 **            dat   = Data for Set up DS3232(1Byte)                   **    
 ************************************************************************/
 
void I2C_WrRTC (unsigned int8 addr,unsigned int8 dat)
{ 
  
   I2C_Start() ;

//---------- Sent Slave address for Write ---------------

   I2C_WrByte(0xD0)       ;  //Send ID Code DS3232+Write (1101 000+0)
 
//------------ Sent address Register --------------
 
   I2C_WrByte(addr)       ;   //Send Address Reg. to DS3232

//------------- Sent data to RTC ---------------

   I2C_WrByte(dat)        ;   //Send Data Control to DS3232
  
   I2C_Stop()             ;
 
 
}


 
/*******************************************************************************
 **                 Function I2C Read  Data 8 bit                             **
 **                                                                           **           
 *******************************************************************************/

 signed int8 I2C_RdByte(void)              
 { 
   signed int8 loop,result=0 ;      

   for(loop = 0; loop < 8; loop++)      //Loop Read data 8-Bit
    {
      result = result << 1         ;   //Shift Result Save (MSB <- LSB)
    
      SDA_IN()                     ;      
      SCL_HI()                     ;   //Strobe Read SDA
             
     if(input(PIN_A0) == 0x01)         //Check Data Read Pin SDA(RA0) is '0' or '1'
       result |= 0x01              ;   //If Read Pin SDA is '1' Give Save Bit Data = 1

      SCL_LO()                     ;   //Next Bit Read

    }
    
  return result                    ;

}



/*************************************************************
 **              Function Read RTC(DS3232) 1 Byte           **
 ** Parameter :                                             **
 **              addr = Register Address for Read            **
 *************************************************************/

 

 signed int8 I2C_RdRTC(unsigned int8 addr)
  {
    signed int8 dat  ;
   
 //-------------- Sent Status Start for write -----------

   I2C_Start()             ;

//------------- Sent Slave Address for Write------------   
    
   I2C_WrByte(0xD0)        ;                //Send ID Code DS3232,Write (1101 000+0)

//------------ Sent address Register --------------

   I2C_WrByte(addr)        ;               //Send Address Reg. to DS3232
 
//---------- New Sent Status Start For Read ----------

   I2C_Start()             ;

//-------------Sent Slave address for Read ------------

  I2C_WrByte(0xD1)         ;               //Send ID Code DS3232 ,Write (1101 000+1)

//------------- Read data 1 Byte ------------
   
   dat = I2C_RdByte()      ;               
  
  

//---------- Sent Status Stop Red ---------------
   I2C_Stop()              ; 
 
   return dat              ;               //Return Data
  }


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  +                                                                            +
  +                      Function Convert data to ASCII                        +
  +                                                                            +
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/


/**************************************************************
 *                                                            *
 *                  Convert BCD Time to ascii                 *
 *                                                            *
 **************************************************************/

 void conv_time(int8 sec,int8 min,int8 hour)
 {
 
   //---------- Convert Second(00-59) to ascii ---------
   ss  = sec >> 4             ;
   ss  = ss+0x30              ; //Secound byte High to ascii

   sec = sec & 0x0F           ;
   sec = sec+0x30             ; //Secound byte Low to ascii
   ss  = (ss << 8)| sec       ; 

   //---------- Convert Minute(00-59) to ascii ---------

   mn  = min >> 4             ;
   mn  = mn+0x30              ;  //Minuted byte High to ascii

   min = min & 0x0F           ;
   min = min+0x30             ;  //Minuted byte Low to ascii
   mn  = (mn << 8)| min       ; 

   //---------- Convert Hour(00-23) to ascii ---------

   hh   = hour >> 4           ;  //Hour byte High to ascii
   hh   = hh+0x30             ;

   hour = hour & 0x0F         ;
   hour = hour+0x30           ;  //Hour byte Low to ascii
   hh   = (hh << 8)| hour     ; 

 }

 
/**************** *********************************************
 *            Convert BCD DATE to ascii                       *
 **************************************************************/

 void conv_date(int8 date,int8 month,int8 year)
 {
  
   //---------- Convert date(00-31) to ascii ---------
   dd   = date >> 4             ;
   dd   = dd+0x30               ;  //Date byte High to ascii

   date = date & 0x0F           ;
   date = date+0x30             ;  //Date byte Low to ascii
   dd   = (dd << 8)| date       ; 

   //---------- Convert Month(00-12) to ascii ---------

   mm    = (month >> 4)& 0x07   ;  //Clear bit Century
   mm    = mm+0x30              ;  //Month byte High to ascii

   month = month & 0x0F         ;
   month = month+0x30           ;  //Month byte Low to ascii
   mm    = (mm << 8)| month     ; 

   //---------- Convert Year(00-99) to ascii ---------

   yy   = year >> 4             ;  //Year byte High to ascii
   yy   = yy+0x30               ;

   year = year & 0x0F           ;
   year = year+0x30             ;  //Year byte Low to ascii
   yy   = (yy << 8)| year       ; 

 }
 
/**************************************************************
 *            Convert hex Temp to ascii                  *
 **************************************************************/

void conv_temp(signed int8 tm,unsigned int8 tf)
{

  //------------- Sign to ascii -------------
   if(tm & 0x80)
     sign = '-'                        ;      //Temp- show sign '-'
    else
     sign = 0x20                       ;      //Temp+ show sign space

  //------------- Temp. Integer hex to ascii -------------

    tm  = tm & 0x7F                    ;      //Clear bit sign
    tem = (tm/10)+0x30                 ;      //Keep Temp byte high to Ascii     
    tem = (tem <<8)|((tm%10)+0x30)     ;      //Keep Temp byte Low to Ascii 
 
  //------------- Temp.(Step=0.25C) fractional hex to ascii -------------

   tf  = tf >>6                        ;     //Shift data bit 7..6  to bit  1..0    
   tf  = tf*25                         ;     //Step temp 25
   dot = (tf/10)+0x30                  ;     //hex to Ascii byte High
   dot = (dot<<8)|((tf%10)+0x30)       ;     //hex to Ascii byte Low
} 

/**********************************************************************
 **                                                                  **
 **                         main Program                             **
 **                                                                  **
 **********************************************************************/
 
void main() 
{
  unsigned int8 sec,min,hor,tf,date,month,year ;
  unsigned int8 tp      ;
  unsigned int16 n      ;

   init_port()          ;
   init_lcd()           ; 

//------------ write string to lcd ----------------
  
  //----------- String mode Time ----------

   write_cmm(0x80|0x00)    ;  //Start address lcd = 0x00  Line 1
   write_data('T')         ;  //Plot data to lcd
   write_data('i')         ;
   write_data('m')         ;
   write_data('e')         ;
   write_data(':')         ;
   write_cmm(0x80|0x07)    ;  //Start address lcd = 0x07  Line 1
   write_data(':')         ;
   write_cmm(0x80|0x0A)    ;  //Start address lcd = 0x0A  Line 1
   write_data(':')         ;

   delay_ms(500);

//--------- Set Time Secound:Minute:Hour to DS3232-------------

   I2C_WrRTC(0x00,0x00)    ;  //Write Second   
   I2C_WrRTC(0x01,0x00)    ;  //Write minute
   I2C_WrRTC(0x02,0x12)    ;  //Write Hour

//--------- Set Date Date:Month:Year to DS3232-------------

   I2C_WrRTC(0x04,0x01)    ;  //Write Date
   I2C_WrRTC(0x05,0x01)    ;  //Write Month
   I2C_WrRTC(0x06,0x12)    ;  //Write year
  
  while(1)
  {
//--------------------- Display Time & Temp ----------------------

    //----------- String mode Temp ----------

   write_cmm(0x80|0x40)    ;  //Start address lcd = 0x40 Line 2
   write_data('T')         ;  //Plot data to lcd
   write_data('e')         ;
   write_data('m')         ;
   write_data('p')         ;
   write_data(':')         ;
   write_cmm(0x80|0x45)    ;  //Start address lcd = 0x45 Line 2
   write_data(0x20)        ;  //Plot data to lcd for sige -
   
   write_cmm(0x80|0x48)    ;  //Start address lcd = 0x48 Line 2
   write_data('.')         ;  //Plot data to lcd
 
   write_cmm(0x80|0x4B)    ;  //Start address lcd = 0x4B Line 2
   write_data(0xD2)        ;  //Plot data to lcd
   write_data('C')         ;  //Plot data to lcd

   for(n=0;n<500;n++)
   {

 //------------ Read Time ss:mm:hh from ds3232 --------------

     sec = I2C_RdRTC(0x00)          ;  //Read Second
     min = I2C_RdRTC(0x01)          ;  //Read minute
     hor = I2C_RdRTC(0x02)          ;  //Read Hour  24 Hour
     conv_time(sec,min,hor)         ;  //Convert sec:min:hor to ascii
   
//------------ Read Temp from ds3232 --------------

     tp = I2C_RdRTC(0x11)           ;  //Read interger + sige temp. 
     tf = I2C_RdRTC(0x12)           ;  //Read fractional(step =0.25 C) temp.
     conv_temp(tp,tf)               ;


//------------ Write Data hh:mm:ss to LCD -------------

     write_cmm(0x80|0x05)    ;  //Start address lcd = 0x05
     write_data(hh>>8)       ;
     write_data(hh)          ;
   
     write_cmm(0x80|0x08)    ;  //Start address lcd = 0x08
     write_data(mn>>8)       ;
     write_data(mn)          ;

     write_cmm(0x80|0x0B)    ;  //Start address lcd = 0x0B
     write_data(ss>>8)       ;
     write_data(ss)          ;

//------------ Write Data Temp to LCD -------------

     write_cmm(0x80|0x45)    ;  //Start address lcd = 0x48
     write_data(sign)        ;  // data Sign

     write_data(tem>>8)      ;  //data integer Temp
     write_data(tem)         ;

     write_cmm(0x80|0x49)    ;  //Start address lcd = 0x4B
     write_data(dot>>8)      ;  //data Fractional Temp
     write_data(dot)         ;

   }

  //-----------------Display Time & Date ------------------

    //----------- String mode Date ----------

   write_cmm(0x80|0x40)    ;  //Start address lcd = 0x40 Line 2
   write_data('D')         ;  //Plot data to lcd
   write_data('a')         ;
   write_data('t')         ;
   write_data('e')         ;
   write_data(':')         ;
   write_cmm(0x80|0x47)    ;  //Start address lcd = 0x47 Line 2
   write_data('/')         ;  //Plot data to lcd for sige -
   
   write_cmm(0x80|0x4A)    ;  //Start address lcd = 0x4A Line 2
   write_data('/')         ;  //Plot data to lcd  

   for(n=0;n<500;n++)
   {
 //------------ Read Time ss:mm:hh from ds3232 --------------

     sec = I2C_RdRTC(0x00)          ;  //Read Second
     min = I2C_RdRTC(0x01)          ;  //Read minute
     hor = I2C_RdRTC(0x02)          ;  //Read Hour  24 Hour
     conv_time(sec,min,hor)         ;  //Convert sec:min:hor to ascii

//------------ Read Date:Mount:Year  from ds3232 --------------

     date  = I2C_RdRTC(0x04)        ;  //Read date (1-31). 
     month = I2C_RdRTC(0x05)        ;  //Read Mount
     year  = I2C_RdRTC(0x06)        ;  //Read year
     conv_date(date,month,year)     ;  //

//------------ Write Data hh:mn:ss to LCD -------------

     write_cmm(0x80|0x05)    ;  //Start address lcd = 0x05
     write_data(hh>>8)       ;
     write_data(hh)          ;
   
     write_cmm(0x80|0x08)    ;  //Start address lcd = 0x08
     write_data(mn>>8)       ;
     write_data(mn)          ;

     write_cmm(0x80|0x0B)    ;  //Start address lcd = 0x0B
     write_data(ss>>8)       ;
     write_data(ss)          ;

//------------ Write Data dd:mm:yy to LCD -------------

     write_cmm(0x80|0x45)    ;  //Start address lcd = 0x45
     write_data(dd>>8)       ;  //Plot data date
     write_data(dd)          ;

     write_cmm(0x80|0x48)    ;  //Start address lcd = 0x48
     write_data(mm>>8)       ;  //Plot data month
     write_data(mm)          ;
  
     write_cmm(0x80|0x4B)    ;  //Start address lcd = 0x4B
     write_data(yy>>8)       ;  //Plot data year
     write_data(yy)          ;   
   }
 }  
}      //End

 


